!> definition of porous media flow model with dual porosity
module modelDoublePorous
  use main_data
  use f_mapping
  use mesh_oper
  use define_state
  use blocks_integ
  use lapack_oper
  use porous_fnc
  use porous_data_module
  use re_analytical

  implicit none

!  public:: Set_f_s_2empty
!  public:: Set_A_s_2empty
!  public:: Set_Ppm_2empty
  public:: Set_R_s_double_porous
  public:: Set_K_sk_double_porous
  public:: Set_Time_Matrix_double_porous
  public:: Exact_Double_Porous
  
  contains

  !> empty convective  terms
  subroutine Set_f_s_2empty(ndimL, nbDim, Qdof, w, f_s, x, ie )
    integer, intent(in) :: Qdof, ndimL, nbDim
    real, dimension(1:Qdof, 1:ndimL), intent(in):: w !state  w in #Qdof nodes
    real, dimension(1:Qdof,1:nbDim,1:ndimL), intent(inout) :: f_s
    real, dimension(1:Qdof,1 :nbDim), intent(in) :: x
    integer, intent(in) :: ie


    f_s = 0.

  end subroutine Set_f_s_2empty

  !> empty convective  terms
  subroutine Set_A_s_2empty(ndimL, nbDim, Qdof, w, A_s, xi, ie)
    integer, intent(in) :: Qdof, nbdim, ndimL
    real, dimension(1:Qdof, 1:ndimL), intent(in):: w !state  w in #Qdof nodes
    real, dimension(1:Qdof,1:nbDim,1:ndimL,1:ndimL), intent(inout) :: A_s
    ! matrices A_s in  -- " --
    real, dimension(1:Qdof,1 :nbDim), intent(in) :: xi
    integer, intent(in) :: ie

    A_s = 0.

  end subroutine Set_A_s_2empty

  !> empty convective  terms
  subroutine Set_Ppm_2empty(ndimL, nbDim, Qdof, w, n, xi, Ppm, one_over_area, elem, ie)
    integer, intent(in) :: Qdof, ndimL, nbDim
    real, dimension(1:Qdof, 1:ndimL), intent(in):: w !state  w in #Qdof nodes
    real, dimension(1:Qdof,1:nbDim,1:ndimL,1:ndimL), intent(inout) :: Ppm
    ! matrices Ppm in  -- " --
    real, dimension(1:Qdof, 1:nbDim), intent(in) :: n   ! outer normal
    real, dimension(1:Qdof, 1:nbDim),intent(in) ::  xi          ! node on the edge?
    real, intent(in), optional :: one_over_area !
    class(element), intent( inout ), optional :: elem !not used
    integer, intent( in ), optional :: ie !not used


    Ppm = 0.

  end subroutine Set_Ppm_2empty


  !> compute viscous fluxes R_s, s=1,2 for the porous media model dual porosity
  !> in integ nodes
  subroutine Set_R_s_double_porous(ndimL, nbDim, iRe, Qdof, w, Dw, Re_1, R_s, xi)
    integer, intent(in) :: ndimL, nbDim, iRe, Qdof
    real, dimension(1:Qdof, 1:ndimL), intent(in):: w !state  w in #Qdof nodes
    real, dimension(1:Qdof, 1:nbDim), intent(in):: xi !physical cooedinates
    real, dimension(1:Qdof, 1:ndimL, 1:nbDim), intent(in):: Dw !state  Dw in #Qdof nodes
    real, dimension(1:iRe, 1:Qdof), intent(in) :: Re_1        ! inverse of Reynolds number
    !real, intent(in) :: Re_1                     ! inverse of Reynolds number
    real, dimension(1:Qdof, 1:nbDim, 1:ndimL), intent(inout) :: R_s
    real, dimension(:, :,:,:,:), allocatable :: K_sk
    integer :: k, l, s

    R_s(:, :, :) = 0.

    !write(*,'(a8,30es12.4)') 'Rs_PM:',xi(1,:), Re_1(1:iRe, 1)
    allocate(K_sk (1:Qdof,1:nbDim,1:nbDim,1:ndimL,1:ndimL) )

    call Eval_Diff_Double_Porous_Coeffs(Qdof, ndimL, &
         w(1:Qdof,1:ndimL), Dw(1:Qdof, 1:ndimL, 1:nbDim), &
         K_sk(1:Qdof,1:nbDim,1:nbDim,1:ndimL, 1:ndimL), &
         Re_1(1:iRe, 1:Qdof), 0, xi(1:Qdof, 1:nbDim) )

    do l=1,Qdof
       do s=1, nbDim
          do k=1, nbDim
       
             R_s(l, s, 1:ndim) =  R_s(l, s, 1:ndim) + &
                  matmul ( K_sk(l, s, k, 1:ndimL, 1:ndimL), Dw(l, 1:ndimL, k) )
          enddo
          !write(*,'(a8, 2i5, 30es12.4)') 'R_S',l,s, R_s(l, s, 1:ndim), Dw(l, 1, 1:2), Dw(l, 2, 1:2)
       enddo
    enddo
    
    deallocate (K_sk)

  end subroutine Set_R_s_double_porous

  !> compute "matrices" 1x1 K_sk, s,k=1,2 for porous media model in integ nodes
  subroutine Set_K_sk_double_porous(ndimL, nbDim, iRe, Qdof, w, Dw, Re_1, K_sk, xi)
    integer, intent(in) :: ndimL, nbDim, iRe, Qdof
    real, dimension(1:Qdof, 1:ndimL), intent(in):: w !state  w in #Qdof nodes
    real, dimension(1:Qdof, 1:nbDim), intent(in):: xi !physical cooedinates
    real, dimension(1:Qdof, 1:ndimL, 1:nbDim), intent(in):: Dw !state  Dw in #Qdof nodes
    real, dimension(1:iRe, 1:Qdof), intent(in) :: Re_1        ! inverse of Reynolds number
    real, dimension(1:Qdof,1:nbDim,1:nbDim,1:ndimL,ndimL), intent(inout) :: K_sk
    integer :: k, l


    call Eval_Diff_Double_Porous_Coeffs(Qdof, ndimL, &
         w(1:Qdof,1:ndimL), Dw(1:Qdof, 1:ndimL, 1:nbDim), &
         K_sk(1:Qdof,1:nbDim,1:nbDim,1:ndimL, 1:ndimL), &
         Re_1(1:iRe, 1:Qdof), 0, xi(1:Qdof, 1:nbDim) )
    

  end subroutine Set_K_sk_double_porous



  !> evaluate "exact" solution, used for the IC and BC
  !> ityp == 1  => returns the exact solution (IC or BC)
  !> ityp == 2  => returns the corresponding right-hand side 
  subroutine Exact_Double_Porous(ityp, Qdof, x, wi, t)
    integer, intent(in) :: ityp
    integer, intent(in) :: Qdof
    real, dimension(1:Qdof, 1:nbDim), intent(in) :: x
    real, dimension(1:Qdof, 1:ndim), intent(out) :: wi
    real, intent(in) :: t
    integer :: imod
    real, dimension(:,:), allocatable :: rr
    real :: u_max, u_min, r_max, tt, t_max
    real :: hinit, width,  length, eps, rf, rft, a0, a1, m, ev, r2, rm, rtt, rt, ev0, ev1
    integer :: l

    allocate( rr(1:2, 1:Qdof) )
    !call Set_Model_Data(x, t, wi, 1)
    imod = state%model%iexact

    !write(*,*)'3de3d imod = ', imod, 'idiff = ', state%model%idiff

    ! imod = 1 damp (hraz) 

    select case (imod)
    case(1)
       if(ityp == 1) then
          wi(:, 1) = 2*x(:, 1)**2 + t
          wi(:, 2) = x(:, 1)**2 + t 
          !wi(:,:) = 1.
          !print*,'###',wi(:, 1:2)
       else if(ityp == 2) then
          wi(:, 1) =  -4. + 1
          wi(:, 2) =  -2. + 1
          !wi(:,:) = 0.

       endif
          
       
    case(2)
       u_min = 0.
       !!u_min = -200.

       wi(:, :) = u_min
       !u_max = 30. - u_min
       u_max = 15. - u_min
       !!u_max = 110. 

       !r_max = 0.05
       r_max = 0.1
       !r_max = 0.5
       !r_max = 1.0
       
       do l=1,Qdof
          ! distance from the left boundary
          if(x(l, 1) <= 0.) then
             rr(1, l) = abs( x(l, 2) )
          elseif( x(l, 2) < 0. .and. x(l, 1) < -15./13 * x(l, 2) ) then
             rr(1,l) = sqrt( dot_product(x(l, 1:2), x(l, 1:2) ) )
          else
             rr(1,l) =  (15./13*x(l, 1) - x(l, 2) ) / sqrt(394./ 169) 
          endif

          if( t > 0) then   ! BC 

             if( x(l, 1) < 15 ) then !  Inlet
                wi(l,1) = u_max
             else                    ! outlet
                wi(l,1) = x(l, 2)    !  h = 0  <==>  H = x_2
             endif

          else   ! modification of the IC
             
             !if( rr(1, l) < r_max) then  
             !   wi( l, 1) = u_max * ( cos(rr(1, l )/r_max *pi ) + 1 )/ 2. + u_min
             !endif

             wi(l, 1) = -0.2 +  x(l, 2)
             if( rr(1, l) < r_max) then  
                !wi( l, 1) = u_max * ( cos(rr(1, l )/r_max *pi ) + 1 )/ 2. + u_min  - 0.2 +  x(l, 2)
                wi(l,1) = u_max * (( r_max - rr(1, l) ) / r_max)**2  &   ! BC
                     + (rr(1,l) /r_max)**2 * ( -0.2 + x(l,2) )           ! IC
             endif
          endif
          
          t_max = -1E-2
          if(t < t_max) then
             tt = sin(t * pi /2 /t_max)
             wi(l, 1) = wi(l, 1) * tt
          endif

          !if(x(l, 2) > 14.25 .and. x(l, 1) < 1000.38) &
          !if(wi(l, 1) < -1) &
          !     write(*, '(a6,2i5, 30es12.4)') 'wBC:',state%nlSolver%iter, l, &
          !     x(l, 1:2), rr(1,l), r_max,t, tt, wi(l, 1)

          !if(rr(1,l ) <= 3*r_max ) &
          !     write(33, *) x(l, 1:2), wi(l, 1), rr(1,l)
       enddo

    case(3) ! test nonlinear case: 2u u_t - (u^2 u_x)_x = 0
       ! C1
       !wi(1:Qdof, 1) = x(1:Qdof, 1)**2 + t

       ! C2
       !wi(1:Qdof, 1) = x(1:Qdof, 1) + t

       ! C3
       !wi(1:Qdof, 1) = exp( t * x(1:Qdof, 1)) 

       ! C4
       !wi(1:Qdof, 1) = exp( t +  x(1:Qdof, 1) )

       ! C5
       !wi(1:Qdof, 1) = (2*  x(1:Qdof, 1) + 1)**3
       wi(1:Qdof, 1) = (2*  x(1:Qdof, 1) + 1)  * ( t + 1)


       !wi(1:Qdof, 1) = x(1:Qdof, 1) 
       !if(t == 0) wi(1:Qdof, 1) = wi(1:Qdof, 1) *0.5
   
       
    case(4:)
 
        stop 'UNKNOWN type in Exact_Double_Porous'

    end select
    deallocate(rr)

  end subroutine Exact_Double_Porous

  !> evaluate "exact" solution, used for the IC and BC
  subroutine Der_Exact_Double_Porous(Qdof, x, Dwi, t)
    integer, intent(in) :: Qdof
    real, dimension(1:Qdof, 1:nbDim), intent(in) :: x
    real, dimension(1:Qdof, 1:ndim, 1:nbDim), intent(out) :: Dwi
    real, intent(in) :: t
    integer :: imod
    real, dimension(:,:), allocatable :: rr
    real :: u_max, u_min, r_max, tt, t_max
    real :: hinit, width,  length, eps, rf, rft, a0, a1, m, ev, evv, r2, rm, rtt, rt, ev0, evx, evy
    integer :: l

    imod = state%model%iexact

    !write(*,*)'3de3d imod = ', imod, 'idiff = ', state%model%idiff

    select case (imod)
    case(10)  !  Barenblatt, porus media flow, Radu et all 2008
       m = state%model%param1       !parameter from *.ini file

       do l=1,Qdof
 
          rm = (m-1.)/(4*m*m)
          r2 = x(l,1)*x(l,1) + x(l,2)*x(l,2)
          rt = (t+1)**(-1./m)

          ev = 1. - rm * r2  * rt


          if(ev < 0) then
             Dwi(l,1,:) = 0.
          else
             ev0 = -2. * rm * rt
             evx = ev0 * x(l,1)
             evy = ev0 * x(l,2)
          
             rf = 1./(t+1)
             a0 = m/(m-1.)
             a1 = a0 - 1.
             
             evv = a0 * ( ev )**a1 * rf
             Dwi(l,1, 1) = evx * evv
             Dwi(l,1, 2) = evy * evv


          endif
          
       enddo
       
       case default
          stop "not available case in subroutine Der_Exact_Porous"
    end select
  end subroutine Der_Exact_Double_Porous

    
  !> evaluation of diffusion coefficients and their derivatives
  !> \f$ K_{s,k}^{i,j},\ s,k=1,2 (\mbox{space dimension}),\ i,j=1,\dots, ndim\f$,
  !> \f$ ider =0 \Rightarrow K(u),\f$ or \f$ ider =0 \Rightarrow K(|\nabla u|),\f$
  !> \f$ ider =1 \Rightarrow \frac{\rm d}{{\rm d} u} K(u) \f$ or
  !> \f$ ider =1 \Rightarrow \frac{\rm d}{{\rm d} |\nabla u|} K(|\nabla u|) \f$
  subroutine Eval_Diff_Double_Porous_Coeffs(Qdof,ndimL, w, Dw, K_sk, Re_1, ider, xi)
    integer, intent(in) :: Qdof ! number of integ nodes
    integer, intent(in) :: ndimL ! number of equations
    real, dimension(1:Qdof, 1:ndimL), intent(in):: w !state  w in #Qdof nodes
    real, dimension(1:Qdof, 1:nbDim), intent(in):: xi !physical cooedinates
    real, dimension(1:Qdof, 1:ndimL, 1:nbDim), intent(in):: Dw !state  Dw in #Qdof nodes
    real, dimension(1:iRe, 1:Qdof), intent(in) :: Re_1        ! inverse of Reynolds number
    real, dimension(1:Qdof,1:nbDim,1:nbDim,1:ndimL, 1:ndimL), intent(inout) :: K_sk
    integer, intent(in) :: ider      ! =0 => K(u), =1 => d K(u)/ d u
    integer :: i, j, imod, nn     ! IMOD
    real :: m, uu, rK, rKp, val1, val2, val3
    real :: viscos, compress, permeab, a0, a1

    imod = state%model%idiff
    !imod = 1    ! Laplace, linear diffusion
    !imod = 2    ! linear diffusion with different coeficients
    !imod = 3    ! nonlinear diffusion, atan
    !imod = 4    ! nonlinear diffusion, atan, anisotrop
    !imod = 5    ! Kacur: degenerate parabolic problem (Eymard, Hilhorst, Vohralik 2006)
    !imod = 6    ! Barenblatt, porus media flow, Radu et all 2008
    !imod = 7    ! NONLINEAR elliptic [Houston, Sulli, Robson 2007]
    !imod = 8    ! NONLINEAR elliptic [Houston, Sulli, Robson 2007] second



    select case (imod)
    case(0)   ! no diffusion
       K_sk = 0.

    case(1)     ! linear diffusion
       call Eval_Diff_Double_Porous_Simple(Qdof,ndimL, &
            w(1:Qdof,1:ndimL), Dw(1:Qdof, 1:ndimL, 1:nbDim), &
            K_sk(1:Qdof, 1:nbDim, 1:nbDim, 1:ndimL, 1:ndimL), &
            Re_1(1:iRe, 1:Qdof), ider, xi(1:Qdof, 1:nbDim) )
       

    ! case(2)     ! non-linear diffusion, original test problem

    !    ! size of the gradient of the head preasure
    !    uu = sqrt(dot_product( Du(1:2) , Du(1:2) )  )

    !    rK = 0.
    !    do i = 1, 3  ! SET THE NUMBER of materials
    !       if(Re_1(i+1) >0.) &
    !            rK = rK +  forch_conduct(u, uu, i, xi(2) ) * Re_1(i+1)

    !       ! if(xi(2) > -0.2 .and. xi(2) <= 0.0 .and. abs(xi(1) -28) <= 0.25) then
    !       !    write(*, '(a8,i5, 16es12.4)') 'conduct:', &
    !       !         i, u, uu, xi(1:2), Re_1(i+1), forch_conduct(u, uu, i, xi(2) ), rK
    !       !    if(i == 3) print*
    !       !    if(i == 3) write(65, *) xi(1:2), rK
    !       ! endif

    !    enddo
       
    !    !write(22, '(30es12.4)' ) xi(1:2), rK

    !    if(ider == 0) then
    !       ! functions
    !       K_sk(1, 1) = rK 
    !       K_sk(2, 2) = K_sk (1,1)

    !       !print*,'#DE#DE#',  rK / compress
    !    else
    !       ! derivatives
    !       !K_sk(1, 1) = rKp / compress
    !       !K_sk(2, 2) = K_sk (1,1)

    !       print* ,'NOT NECESSARY'
    !    endif


    case(2:)
       stop 'UNKNOWN TYPE in Eval_Diff_Double_Porous_Coeffs'

    end select

  end subroutine Eval_Diff_Double_Porous_Coeffs


  !> the simples linear diffusion, two-equation heat equation
  subroutine Eval_Diff_Double_Porous_Simple(Qdof, ndimL, w, Dw, K_sk, Re_1, ider, xi)
    integer, intent(in) :: ndimL ! number of integ nodes
    integer, intent(in) :: Qdof  ! number of integ nodes
    real, dimension(1:Qdof, 1:ndimL), intent(in):: w !state  w in #Qdof nodes
    real, dimension(1:Qdof, 1:nbDim), intent(in):: xi !physical cooedinates
    real, dimension(1:Qdof, 1:ndimL, 1:nbDim), intent(in):: Dw !state  Dw in #Qdof nodes
    real, dimension(1:iRe, 1:Qdof), intent(in) :: Re_1        ! inverse of Reynolds number
    real, dimension(1:Qdof, 1:nbDim, 1:nbDim, 1:ndimL, 1:ndimL), intent(inout) :: K_sk
    integer, intent(in) :: ider      ! =0 => K(u), =1 => d K(u)/ d u
    integer :: i, j, imod, nn     ! IMOD
    real :: m, uu, rK, rKp, val1, val2, val3
    real :: viscos, compress, permeab, a0, a1

    K_sk = 0.
    
    do i=1, Qdof
       K_sk(i, 1, 1, 1, 1) = Re_1(1,i )
       K_sk(i, 1, 1, 2, 2) = Re_1(1,i )

       K_sk(i, 2, 2, 1, 1) = Re_1(1,i )
       K_sk(i, 2, 2, 2, 2) = Re_1(1,i )
    enddo
    
    

  end subroutine Eval_Diff_Double_Porous_Simple
  
  !> evaluation of the water content in integ node
  subroutine Eval_double_water_contentVec(Qdof, u, wat_cont, Re_1, xi, only_wc_in)
    integer, intent(in) :: Qdof
    real, dimension(1:Qdof, 1:ndim), intent(in) :: u            ! solution
    real, dimension(1:Qdof, 1:nbDim), intent(in) :: xi ! physical coordinate
    real, dimension(1:Qdof, 1:ndim), intent(inout) :: wat_cont ! output diffusion matrix
    real, dimension(1:Qdof, 1:iRe-1), intent(in) :: Re_1         ! viscosity
    logical, intent(in), optional :: only_wc_in
    integer :: i

    do i=1,Qdof 
       call Eval_double_water_content(u(i, 1:ndim), wat_cont(i, 1:ndim), Re_1(i,1:iRe-1), &
            xi(i,1:nbDim), only_wc_in)
    enddo
  end subroutine Eval_double_water_contentVec
    
    !> evaluation of the water content in integ node
  subroutine Eval_double_water_content(u, wat_cont, Re_1, xi, only_wc_in)
    real, dimension(1:ndim), intent(in) :: u            ! solution
    real, dimension(1:nbDim), intent(in) :: xi ! physical coordinate
    real, dimension(1:ndim), intent(inout) :: wat_cont ! output diffusion matrix
    real, dimension(1:iRe-1), intent(in) :: Re_1         ! viscosity
    logical, intent(in), optional :: only_wc_in
    real :: F0, F1, F2, F3
    real, dimension(:), pointer :: TK, XD, YD, Y
    integer :: i, j,  nn     ! IMOD
    real :: m, uu, rK, rKp, val1, val2, val3, h, s
    real :: viscos, compress, permeab, a0, a1, theta_loc, integ, t, integ1, eps, ax, bx
    integer :: l1, l2  , ii, itime
    logical ::  only_wc

    only_wc = .false.  ! only water content without storativity
    if(present(only_wc_in) ) then
       only_wc = only_wc_in
    endif
    
    
    itime = state%model%itime
    !itime = 3    ! Van Genuchten relation

    wat_cont = 0.
    select case (itime)
    case(1)     ! heat equation
       wat_cont(1) = u(1)
       wat_cont(2) = u(2)
       
    case(2:5)     ! Van Genuchten relation

       
       wat_cont = 0.
       do i = 1, 3  ! SET THE NUMBER of materials

          if(water_cubic_interpol) then
             nn = soilpar(i)%num_vol
             TK => soilpar(i)%rec_spline(1, 1: nn+1)
             XD => soilpar(i)%rec_spline(2, 1: nn+1)
             YD => soilpar(i)%rec_spline(3, 1: nn+1)
             
             Y => soilpar(i)%volume(0 : nn)
          endif
          
          
          ! version with \theta(h) only
          !if(Re_1(i+1) >0.) &
          !     wat_cont = wat_cont +  vangen(u,  i, xi(2) ) * Re_1(i+1)

          ! version with \theta(h) and adding of the storativity
          if(Re_1(i) >0.) then

             theta_loc = vangen(u(1),  i, xi(2) )

             h = u(1) - xi(2)

             integ = 0.
             if(.not. only_wc) then

                if(h >= 0) then
                   integ = (soilpar(i)%volume( 0 ) / soilpar(i)%ths + h ) * soilpar(i)%Ss 
                   
                else
                   
                   if(abs(h * soilpar(i)%h_int) + 1 >  soilpar(i)%num_vol) then ! approximation of "infinity"
                      integ =  soilpar(i)%volume( soilpar(i)%num_vol)
                      
                   else
                      !print*,'###E#D#D#:',i, h, soilpar(i)%h_int 
                      l1 = int (abs(h * soilpar(i)%h_int ) )
                      l2 = l1 + 1
                      
                      t = abs( h  * soilpar(i)%h_int - l1  )
                      
                      ! linear interpolation of the tabular values
                      integ =  soilpar(i)%volume( l2 ) * t + soilpar(i)%volume( l1 ) * ( 1- t)
                      
                      ! spline interpolation of the tabular values
                      if(water_cubic_interpol) then
                         ii = l2
                         s = t  * TK(i)
                         F0=2.0*(S/TK( ii ))**3-3.0*(S/TK( ii ))**2+1
                         F1=-2.0*(S/TK( ii ))**3+3.0*(S/TK( ii ))**2
                         F2=S**3/TK( ii )**2-2.0*S**2/TK( ii )+S
                         F3=S**3/TK( ii )**2-S**2/TK( ii )
                         !XI=F0*X( ii )+F1*X( ii+1 )+F2*XD( ii )+F3*XD( ii+1 )
                         integ = F0*Y( ii )+F1*Y( ii+1 )+F2*YD( ii )+F3*YD( ii+1 )
                      endif
                      !write(98,*) h,  integ, integ1, t
                   endif
                   
                   integ  = integ * soilpar(i)%Ss / soilpar(i)%ths
                   !!integ1  = integ1 * soilpar(i)%Ss / soilpar(i)%ths
                   
                   !write(31,'(a4, i5, 7es12.4, 5i5)')  'conT:', i, h, integ, theta_loc,  soilpar(i)%h_int !, &
                   !  abs(h * soilpar(i)%h_int), t,  integ,  theta_loc + integ, l1 , l2
                   
                endif  ! if(h >= 0)
             endif  ! if(.not. only_wc)
             !stop "333kihj"

             ! composition of all materials
             wat_cont = wat_cont +  ( theta_loc + integ) * Re_1(i)

             !if( h > 0) &
             !     write(*,'(a4, i5, l3, 71es12.4)')  &
             !     'conT:', i, only_wc, u, xi(2), h, theta_loc, integ,Re_1(i), wat_cont

             
          endif  ! if(Re_1(i) > 0.)
               
          ! if(xi(2) > -0.2 .and. xi(2) <= 0.0 .and. abs(xi(1) -28) <= 0.25) then
          !    write(*, '(a8,i5, 16es12.4)') 'conduct:', &
          !         i, u, uu, xi(1:2), Re_1(i+1), forch_conduct(u, uu, i, xi(2) ), rK
          !    if(i == 3) print*
          !    if(i == 3) write(65, *) xi(1:2), rK
          ! endif

       enddo


    case(6)  ! Gardner's model

       wat_cont = 0.
       do i = 1, 3  ! SET THE NUMBER of materials

          theta_loc = gardner_wc(u(1),  i, xi(2) )
          
          wat_cont = wat_cont +  theta_loc  * Re_1(i)
       enddo

    case(7:8)     ! Van Genuchten relation, one material, No 4

       wat_cont = 0.
       do i = 4, 4  ! SET THE NUMBER of materials

          if(water_cubic_interpol) then
             nn = soilpar(i)%num_vol
             TK => soilpar(i)%rec_spline(1, 1: nn+1)
             XD => soilpar(i)%rec_spline(2, 1: nn+1)
             YD => soilpar(i)%rec_spline(3, 1: nn+1)
             
             Y => soilpar(i)%volume(0 : nn)
          endif
          
          
          ! version with \theta(h) only
          !if(Re_1(i+1) >0.) &
          !     wat_cont = wat_cont +  vangen(u,  i, xi(2) ) * Re_1(i+1)

          ! version with \theta(h) and adding of the storativity
          !if(Re_1(i) >0.) then

          theta_loc = vangen(u(1),  i, xi(2) )

          h = u(1) - xi(2)
          
          integ = 0.
          if(.not. only_wc) then
             
             if(h >= 0) then
                integ = (soilpar(i)%volume( 0 ) / soilpar(i)%ths + h ) * soilpar(i)%Ss 
                
             else
                
                if(abs(h * soilpar(i)%h_int) + 1 >  soilpar(i)%num_vol) then ! approximation of "infinity"
                   integ =  soilpar(i)%volume( soilpar(i)%num_vol)
                   
                else
                   !print*,'###E#D#D#:',i, h, soilpar(i)%h_int 
                   l1 = int (abs(h * soilpar(i)%h_int ) )
                   l2 = l1 + 1
                   
                   t = abs( h  * soilpar(i)%h_int - l1  )
                   
                   ! linear interpolation of the tabular values
                   integ =  soilpar(i)%volume( l2 ) * t + soilpar(i)%volume( l1 ) * ( 1- t)
                   
                   ! spline interpolation of the tabular values
                   if(water_cubic_interpol) then
                      ii = l2
                      s = t  * TK(i)
                      F0=2.0*(S/TK( ii ))**3-3.0*(S/TK( ii ))**2+1
                      F1=-2.0*(S/TK( ii ))**3+3.0*(S/TK( ii ))**2
                      F2=S**3/TK( ii )**2-2.0*S**2/TK( ii )+S
                      F3=S**3/TK( ii )**2-S**2/TK( ii )
                      !XI=F0*X( ii )+F1*X( ii+1 )+F2*XD( ii )+F3*XD( ii+1 )
                      integ = F0*Y( ii )+F1*Y( ii+1 )+F2*YD( ii )+F3*YD( ii+1 )
                   endif
                   !write(98,*) h,  integ, integ1, t
                endif
                
                integ  = integ * soilpar(i)%Ss / soilpar(i)%ths
                !!integ1  = integ1 * soilpar(i)%Ss / soilpar(i)%ths
                
                !write(31,'(a4, i5, 7es12.4, 5i5)')  'conT:', i, h, integ, theta_loc,  soilpar(i)%h_int !, &
                !  abs(h * soilpar(i)%h_int), t,  integ,  theta_loc + integ, l1 , l2
                
             endif  ! if(h >= 0)
          endif  ! if(.not. only_wc)
          !stop "333kihj"
          
          ! composition of all materials
          wat_cont = wat_cont +  ( theta_loc + integ) !* Re_1(i)
          
          !if( h > 0) &
          !     write(*,'(a4, i5, l3, 71es12.4)')  &
          !     'conT:', i, only_wc, u, xi(2), h, theta_loc, integ,Re_1(i), wat_cont
          
             
          !!endif  ! if(Re_1(i) > 0.)
               
          ! if(xi(2) > -0.2 .and. xi(2) <= 0.0 .and. abs(xi(1) -28) <= 0.25) then
          !    write(*, '(a8,i5, 16es12.4)') 'conduct:', &
          !         i, u, uu, xi(1:2), Re_1(i+1), forch_conduct(u, uu, i, xi(2) ), rK
          !    if(i == 3) print*
          !    if(i == 3) write(65, *) xi(1:2), rK
          ! endif

       enddo  ! do i=4,4

    case(9:)
       print*, 'UNKNOWN TYPE ', itime, '  in  Eval_double_water_content'
       stop
    end select

    !write(*,'(a4, i5, l3, 71es12.4)')  &
    !     'conT:', i, only_wc, u, xi(2), h, theta_loc, integ, wat_cont

  end subroutine Eval_double_water_content



  !> compute the matrix in front of the time derivative term in integ nodes
  subroutine Set_Time_Matrix_double_porous(elem, ndimL,  Qdof, wi, xi, TA, wR)
    class(element), intent (inout) :: elem
    integer, intent(in) :: ndimL,  Qdof
    real, dimension(1:Qdof, 1:ndimL), intent(in):: wi  !state  w in #Qdof nodes
    real, dimension(1:Qdof, 1:2+iRe), intent(in):: xi  !coodinates + participation to components
    real, dimension(1:Qdof, 1:ndimL, 1:ndimL), intent(inout) :: TA ! output matrix 
    !!real, dimension(1:ndimL, 1:elem%dof_plus), optional :: wR
    real, dimension(1:ndimL, 1:elem%dof), optional :: wR
    real, dimension(:, :), allocatable :: ww
    real :: rK, xc(2), m, eps, ax, bx, u
    integer :: i, l, dof, itime
    real, dimension(:,:), pointer :: phi

    TA = 0.

    itime =  state%model%itime
    
    select case (itime)
 
    case(  1 ) 
       do l=1, Qdof
          TA(l,1, 1) = 1.
          TA(l,2, 2) = 1.
       enddo
       
    case(2)
       do l=1, Qdof
          rK = 0.
          do i = 1, 3  ! SET THE NUMBER of materials
             ! elem%xi(0, l, 2+i) is the coefficent of affiliation to  i-th material component
             if(xi( l, 2+i) > 0.) & 
                  rK = rK +  capacity(wi(l,1), i, xi(l, 2) ) * xi(l, 2+i)
             !rK = rK +  capacity(wi(l,1), i, elem%xi(0,l, 2) ) * elem%xi(0, l, 2+i)

             !if(i == 3 .and.  elem%i == 1297 ) then
             !!if( abs(xi(l, 1) - 12.15042) < 1E-3 .and. abs(xi(l, 2) - 1.052989)<1E-3) then
             !   write(*, '(a8,3i5, 60es12.4)') 'conduct:', &
             !        elem%i, l, i, rk,  capacity(wi(l,1), i, xi(l, 2) ) * xi(l, 2+i), &
             !        wi(l,1) - xi(l, 2),   wi(l,1), xi(l, 1:2)  , xi(l, 2+i)
             !endif


             !write(*, '(a8,i5, 6es12.4)') 'conduct:', i, u, uu, xi(2), Re_1(i+1), rK
          enddo
          !if(elem%i == 1297 ) print*

          TA(l,1, 1) = rK
          TA(l,2, 2) = rK
          !if(rk < 0.) &
          !write(21, '(23es12.4)' ) elem%xi(0,l,1:2), rK,  elem%xi(0,l,3:)

          !if(state%time%iter >= 1) then
          !   write(66,'(30es12.4)')  &
          !        elem%xi(0, l, 1:2), rK, wi(l, 1),elem%xi(0, l, 2+i)
          !   if(elem%i == grid%nelem) stop '9ue93jdo3dmzd39u393i'
          !endif

       enddo

    case(3:)
       print*,'Case ',itime,' is not implemented'
       print*, 'in subroutine  Set_Time_Matrix_double_porous (model2porous.f90)'
       stop
    end select


    !write(*,'(i5, 30es12.4)') state%time%iter, xi(1, 1:2), TA(1,1,1), xi(1, 3:2+iRe)

    ! xc(1) = 10.5; xc(2) =0.25
    ! do l=1, Qdof
    !    !if( abs( elem%xi(0, l, 1) -12.) < 3 .and. abs( elem%xi(0, l, 2) -0.) < 3) &
    !    !if(TA(l,1,1) < 0.) &
    !    if( dot_product(elem%xc- xc , elem%xc- xc) < 2.) &
    !        write(62,'(30es12.4)') elem%xi(0,l, 1:2), TA(l,1,1)
    ! enddo

    !if(elem%i <= 20) &
    !write(*,'(a8, i5, 30es14.6)') 'PQRE#4', elem%i, wi(1:3,1),  TA(1:3,1,1)

    if( present(wR) ) then

       ! ! projection of the integ nodes given functions onto polynomial function
       !dof = elem%dof_plus
       dof = elem%dof
       !dof = DoFtriang( max(0, elem%deg - 1) )
       allocate(ww(1:dof, 1:2) )
       
       !!if( abs( TA(1, 1, 1) - TA(2, 1,1 ) ) > 0.15 ) then
       !if(elem%i == 843) then
       !write(*,'(a12, 2i5, 300es12.4)') ' TA:', elem%i, Qdof,  TA(1:Qdof, 1, 1)
       !endif

       if(Qdof /= elem%Qdof) then
          print*,'Trouble in modelPorous.f90 9u39uj93'
          print*,' the following   call IntegrateVectorB(   is inconsitent'
          print*,' Probably mixture of L_rule and V_rule'
       endif
       
       ww(:,:) = 0.
       call IntegrateVectorB(elem, dof, TA(1:Qdof, 1, 1), ww(1:dof, 1) )
       
       !write(*,'(a12, 2i5, 300es12.4)') ' w,phi:', elem%i, dof,  ww(1:dof, 1)
       
       !do l=1,dof
       !   ww(l, 2) = dot_product(elem%MassInv%Mb(l,1:dof), ww(1:dof, 1) )
       !enddo
       ww(1:dof, 2) = ww(1:dof, 1)
       
       call SolveLocalMatrixProblem(dof, elem%mass%Mb(1:dof, 1:dof), 1, ww(1:dof, 2))
       
       wR = 0.
       wR(1, 1:dof) = ww(1:dof, 2)
    

       !write(*,'(a12, 2i5, 300es12.4)') 'M^-1 w,phi:', elem%i, dof,  ww(2, 1:dof)
       
       !phi => state%space%V_rule(elem%Qnum)%phi(1:dof,1:Qdof)
       !TA(1:Qdof, 1, 1) = matmul(ww(1:dof, 2) , phi(1:dof,1:Qdof) )
       
       
       deallocate(ww)

    endif

    ! scaling factor for porous media flow
    elem%CC(Cti) = maxval( TA )
    
    ! do l=1, Qdof
    !    !if( abs( elem%xi(0, l, 1) -12.) < 3 .and. abs( elem%xi(0, l, 2) -0.) < 3) &
    !   !if(TA(l,1,1) < 0.) &

    !    if( dot_product(elem%xc- xc , elem%xc- xc) < 2.) &
    !        write(64,'(30es12.4)') elem%xi(0,l, 1:2), TA(l,1,1)
    ! enddo


    !if( abs( TA(1, 1, 1) - TA(2, 1,1 ) ) > 0.15 ) then
    !if(elem%i == 843) then
    !   write(*,'(a12, 2i5, 300es12.4)') ' TA:', elem%i, dof,  TA(1:Qdof, 1, 1)
    !   print*,"_____________________",  abs( TA(1, 1, 1) - TA(2, 1,1 ) )
    !   !stop '9e39ud93o'
    !endif

  end subroutine Set_Time_Matrix_double_porous


  
end module modelDoublePorous

